unit CompPolyline;

interface

uses Classes, Forms, Graphics, Windows,
     Face;

type
  tPointList = class
  protected
    fOwner: tComponent;
  public
    PointArray: tPointArray;
    constructor Create(aOwner: tComponent);
    destructor Destroy; override;
    procedure DrawPolyline;
    procedure DrawPolygon;
    function GetRgn: hRgn;
    function OnItem(P: tPoint): Boolean;
    procedure Update;
  end;

  tPolyline = class(tDrawComponent)
  private
    PointList: tPointList;
  public
    Closed, Filled: Boolean;
    constructor CreateBy(aOwner: tComponent); override;
    constructor CreateByPoint(aOwner: tComponent; aPoint: tPoint); override;
    destructor Destroy; override;
    procedure AddPoint(aPoint: tPoint); override;
    procedure GetExtent(var aMin, aMax: tPoint); override;
    function GetRgn: hRgn; override;
    procedure Paint; override;
    procedure PutGrips; override;
    procedure RemoveLast; override;
    procedure SetPropList; override;
    procedure SizeTo(X, Y: Integer); override;
    procedure Update; override;
  end;

var dciPolyline: Integer;

implementation

uses Dialogs, SysUtils,
     GDIPAPI, GDIPOBJ;

constructor tPointList.Create(aOwner: tComponent);
begin
  inherited Create;
  fOwner:= aOwner;
end;

destructor tPointList.Destroy;
begin
  if Assigned(PointArray) then PointArray.Destroy;
  inherited;
end;

procedure tPointList.DrawPolyline;
begin
  Update;
  PolyLine(DrawFace.Canvas.Handle, PointArray.pA^, PointArray.Count);
end;

procedure tPointList.DrawPolygon;
begin
  Update;
  Polygon(DrawFace.Canvas.Handle, PointArray.pA^, PointArray.Count);
end;

function tPointList.GetRgn: hRgn;
var i, DD: Integer;
    P1, P2: Windows.tPoint;
    P1F: TGPPointF;
    R1, R2: TGPRegion;
    AA: Single;
    MM: TGPMatrix;
    myGraphic: TGPGraphics;
begin
  if tPolyline(fOwner).Closed and tPolyline(fOwner).Filled then
  begin
    Result:= CreatePolygonRgn(PointArray.pA^, PointArray.Count, WINDING);
    Exit;
  end;
  try
    R1:= TGPRegion.Create;
    for i:= 1 to PointArray.Count- 1 do
    begin
      P1:= PointArray.GetElem(i- 1);
      P1F:= MakePoint(P1.X*1.0, P1.Y*1.0);
      P2:= PointArray.GetElem(i);
      DD:= Round(Dist(P1, P2));
      AA:= RadToAng(Ang(P1, P2));
      MM:= TGPMatrix.Create;
      MM.Translate(P1F.X, P1F.Y, MatrixOrderAppend);
      MM.RotateAt(AA, P1F, MatrixOrderAppend);
      if i= 1 then
      begin
        R1.Free;
        R1:= TGPRegion.Create(MakeRect(0, -2, DD, 4));
        R1.Transform(MM);
      end else
      begin
        R2:= TGPRegion.Create(MakeRect(0, -2, DD, 4));
        R2.Transform(MM);
        R1.Union(R2);
        R2.Free;
      end;
      MM.Free;
    end;
    if tPolyline(fOwner).Closed then
    begin
      P1:= PointArray.GetElem(PointArray.Count- 1);
      P1F:= MakePoint(P1.X*1.0, P1.Y*1.0);
      P2:= PointArray.GetElem(0);
      DD:= Round(Dist(P1, P2));
      AA:= RadToAng(Ang(P1, P2));
      MM:= TGPMatrix.Create;
      MM.Translate(P1F.X, P1F.Y, MatrixOrderAppend);
      MM.RotateAt(AA, P1F, MatrixOrderAppend);
      R2:= TGPRegion.Create(MakeRect(0, -2, DD, 4));
      R2.Transform(MM);
      R1.Union(R2);
      R2.Free;
      MM.Free;
    end;
    myGraphic := TGPGraphics.Create(DrawFace.Canvas.Handle);
    Result:= R1.GetHRGN(myGraphic);
    myGraphic.Free;
    R1.Free;
  except
    Result:= 0;
  end;
end;

function tPointList.OnItem(P: Windows.tPoint): Boolean;
var R: hRgn;
begin
  Result:= False;
  if PointArray.Count<3 then Exit;
  R:= GetRgn;
  if PtInRegion(R, P.X, P.Y) then Result:= True;
  DeleteObject(R);
end;

procedure tPointList.Update;
var i: Integer;
    List: tList;
    PP: Windows.tPoint;
begin
  List:= tList.Create;
  for i:= 0 to fOwner.ComponentCount- 1 do
    if (fOwner.Components[i] is tDrawPoint) then
      List.Add(fOwner.Components[i]);
  if Assigned(PointArray) then PointArray.Free;
  PointArray:= tPointArray.Create(List.Count);
  for i:= 0 to List.Count- 1 do
  begin
    PP:= tDrawPoint(List.Items[i]).Value;
    PointArray.SetElem(i,
      Point(Round(PP.X* DrawFace.ZoomFactor)- DrawFace.Shift.X,
            Round(PP.Y* DrawFace.ZoomFactor)- DrawFace.Shift.Y));
  end;
  List.Free;
end;

///////////////////////  Polyline

constructor tPolyline.CreateBy(aOwner: tComponent);
begin
  ImageIndex:= 11;
  inherited CreateByBasicName(aOwner, 'Polyline');
  PointList:= tPointList.Create(Self);
  Color:= clMoneyGreen;
  LineColor:= clBlack;
  LineWidth:= 1;
  Closed:= False;
  className:= 'Polyline';
  CreatingByAddPoint:= True;
end;

constructor tPolyline.CreateByPoint(aOwner: tComponent; aPoint: Windows.tPoint);
begin
  CreateBy(aOwner);
  with tDrawPoint.CreateBy(Self) do Value:= aPoint;
  with tDrawPoint.CreateBy(Self) do Value:= aPoint;
end;

destructor tPolyline.Destroy;
begin
  PointList.Free;
  inherited;
end;

procedure tPolyline.AddPoint(aPoint: Windows.tPoint);
begin
  with tDrawPoint.CreateBy(Self) do Value:= aPoint;
  PointList.Update;
end;

procedure tPolyline.GetExtent(var aMin, aMax: Windows.tPoint);
var i: Integer;
begin
  for i:= 0 to ComponentCount- 1 do
    tDrawComponent(Components[i]).GetExtent(aMin, aMax);
end;

function tPolyline.GetRgn: hRgn;
begin
  Result:= PointList.GetRgn;
end;

procedure tPolyline.Paint;
begin
  if not Assigned(PointList.PointArray) then Update;
  with DrawFace.Canvas do
  begin
    Brush.Color:= Color;
    if (Closed and Filled) then Brush.Style:= bsSolid
                           else Brush.Style:= bsClear;
    Pen.Mode:= pmCOPY;
    if DrawFace.Printing then Pen.Color:= clBlack
                         else Pen.Color:= LineColor;
    Pen.Width:= Round(LineWidth* DrawFace.ZoomFactor);
    if Pen.Width< 1 then Pen.Width:= 1;
  end;
  if Closed then PointList.DrawPolygon
            else PointList.DrawPolyline;
  inherited;
end;

procedure tPolyline.PutGrips;
var i: Integer;
begin
  for i:= 0 to ComponentCount- 1 do
    if Components[i] is tDrawPoint then
      (Components[i] as tDrawPoint).PutGrips;
end;

procedure tPolyline.RemoveLast;
begin
  DrawFace.Tree.DeleteComponent(Components[ComponentCount-1]);
  Update;
end;

procedure tPolyline.SetPropList;
begin
  inherited;
  with PropList do
  begin
    Kill('Left');
    Kill('Top');
    Kill('Width');
    Kill('Height');
    Add(tBooleanProp.CreateBooleanProp(@Closed, 'Closed'));
    Add(tBooleanProp.CreateBooleanProp(@Filled, 'Filled'));
  end;
end;

procedure tPolyline.SizeTo(X, Y: Integer);
begin
  tDrawPoint(Components[ComponentCount- 1]).MoveTo(X, Y);
  PointList.Update;
end;

procedure tPolyline.Update;
begin
  PointList.Update;
  inherited;
end;

initialization

  //dci -> DrawComponentIndex

  dciPolyline:= Face.DrawCompItemList.Add(tDrawCompItem.Create('Polyline', tPolyline));

end.
